from flask import Flask, render_template, request, jsonify, send_file, flash, redirect, url_for
import requests
import os
import uuid
from werkzeug.utils import secure_filename
import time
import io
from PIL import Image
import base64

app = Flask(__name__)
app.secret_key = 'your-secret-key-here-change-this'  # Change this to a secure secret key

# Configuration
UPLOAD_FOLDER = 'uploads'
RESULTS_FOLDER = 'results'
ALLOWED_EXTENSIONS = {'png', 'jpg', 'jpeg', 'gif'}
MAX_FILE_SIZE = 16 * 1024 * 1024  # 16MB

# Create directories if they don't exist
os.makedirs(UPLOAD_FOLDER, exist_ok=True)
os.makedirs(RESULTS_FOLDER, exist_ok=True)

# API configuration
RAPIDAPI_URL = "https://ai-face-swap2.p.rapidapi.com/public/process/base64/base64"
RAPIDAPI_KEY = "bb3aef5a62mshb43817c3f3f3638p16af3ajsn4bd2732a2903"

# MagicHour API configuration (fallback)
MAGICHOUR_UPLOAD_URL = "https://api.magichour.ai/v1/files/upload-urls"
MAGICHOUR_API_URL = "https://api.magichour.ai/v1/face-swap-photo"
# Get API key from environment variable, or use the hardcoded one as fallback
MAGICHOUR_API_KEY = os.getenv('MAGICHOUR_API_KEY', 'mhk_live_r36p6tjr8dpZkF58t952xAYV8LMYnxN8oAgVfTj7NCZudRzdgWrAdtow0zMpPwfFgFysC5Amo3mJKLnn')

def allowed_file(filename):
    return '.' in filename and filename.rsplit('.', 1)[1].lower() in ALLOWED_EXTENSIONS

def create_face_swap_rapidapi(source_image_path, target_image_path):
    """Call RapidAPI Face Swap service"""
    try:
        # Convert images to base64
        with open(source_image_path, 'rb') as f:
            source_b64 = base64.b64encode(f.read()).decode('utf-8')
        
        with open(target_image_path, 'rb') as f:
            target_b64 = base64.b64encode(f.read()).decode('utf-8')
        
        headers = {
            "x-rapidapi-key": RAPIDAPI_KEY,
            "x-rapidapi-host": "ai-face-swap2.p.rapidapi.com",
            "Content-Type": "application/x-www-form-urlencoded"
        }
        
        # Prepare the payload with base64 images
        payload = {
            "source_image": source_b64,
            "target_image": target_b64
        }
        
        response = requests.post(RAPIDAPI_URL, data=payload, headers=headers)
        
        if response.status_code == 200:
            result = response.json()
            if result.get('message') == 'Processing successful':
                return {
                    "success": True,
                    "image_base64": result.get('image_base64'),
                    "processing_time": result.get('processing_time', 0),
                    "service": "rapidapi"
                }
            else:
                return {"error": f"Processing failed: {result.get('message', 'Unknown error')}"}
        else:
            return {"error": f"RapidAPI request failed with status {response.status_code}: {response.text}"}
            
    except Exception as e:
        return {"error": f"RapidAPI request failed: {str(e)}"}

def upload_image_to_magichour(image_path):
    """Upload an image to MagicHour and return the asset path"""
    if not MAGICHOUR_API_KEY:
        return None, "MagicHour API key not configured"
    
    try:
        # Get file extension
        filename = os.path.basename(image_path)
        extension = filename.rsplit('.', 1)[1].lower()
        
        # Step 1: Get upload URL
        headers = {
            'Authorization': f'Bearer {MAGICHOUR_API_KEY}',
            'Content-Type': 'application/json'
        }
        
        upload_request = {
            "items": [
                {
                    "extension": extension,
                    "type_": "image"
                }
            ]
        }
        
        response = requests.post(MAGICHOUR_UPLOAD_URL, json=upload_request, headers=headers)
        
        if response.status_code != 200:
            return None, f"Failed to get upload URL: {response.status_code} - {response.text}"
        
        upload_data = response.json()
        upload_url = upload_data['items'][0]['upload_url']
        file_path = upload_data['items'][0]['file_path']
        
        # Step 2: Upload the file
        with open(image_path, 'rb') as f:
            image_data = f.read()
        
        upload_response = requests.put(upload_url, data=image_data)
        
        if upload_response.status_code not in [200, 201]:
            return None, f"Failed to upload file: {upload_response.status_code}"
        
        return file_path, None
        
    except Exception as e:
        return None, f"Upload failed: {str(e)}"

def create_face_swap_magichour(source_image_path, target_image_path):
    """Call MagicHour API to create face swap (fallback)"""
    if not MAGICHOUR_API_KEY:
        return {"error": "MagicHour API key not configured"}
    
    try:
        # Upload images to MagicHour
        source_asset_path, source_error = upload_image_to_magichour(source_image_path)
        if source_error:
            return {"error": f"Source image upload failed: {source_error}"}
        
        target_asset_path, target_error = upload_image_to_magichour(target_image_path)
        if target_error:
            return {"error": f"Target image upload failed: {target_error}"}
        
        headers = {
            'Authorization': f'Bearer {MAGICHOUR_API_KEY}',
            'Content-Type': 'application/json'
        }
        
        payload = {
            "assets": {
                "face_mappings": [
                    {
                        "new_face": source_asset_path,
                        "original_face": target_asset_path,
                    }
                ],
                "face_swap_mode": "all-faces",
                "source_file_path": source_asset_path,
                "target_file_path": target_asset_path,
            },
            "name": f"Face Swap - {time.strftime('%Y-%m-%d %H:%M:%S')}"
        }
        
        response = requests.post(MAGICHOUR_API_URL, json=payload, headers=headers)
        
        if response.status_code == 200:
            result = response.json()
            return {
                "success": True,
                "id": result.get("id"),
                "credits_charged": result.get("credits_charged", 5),
                "service": "magichour"
            }
        else:
            return {
                "error": f"MagicHour API request failed with status {response.status_code}: {response.text}"
            }
            
    except requests.exceptions.JSONDecodeError as e:
        return {"error": f"Invalid JSON response from MagicHour API. This might indicate an authentication issue."}
    except Exception as e:
        return {"error": f"MagicHour request failed: {str(e)}"}

def create_face_swap(source_image_path, target_image_path):
    """Main face swap function - tries RapidAPI first, then MagicHour as fallback"""
    print(f"Attempting face swap with RapidAPI first...")
    
    # Try RapidAPI first
    rapidapi_result = create_face_swap_rapidapi(source_image_path, target_image_path)
    
    if rapidapi_result.get("success"):
        print("RapidAPI face swap successful!")
        return rapidapi_result
    
    print(f"RapidAPI failed: {rapidapi_result.get('error')}")
    print("Falling back to MagicHour API...")
    
    # Fallback to MagicHour
    magichour_result = create_face_swap_magichour(source_image_path, target_image_path)
    
    if magichour_result.get("success"):
        print("MagicHour API face swap successful!")
        return magichour_result
    
    print(f"MagicHour API also failed: {magichour_result.get('error')}")
    
    # Both services failed
    return {
        "error": f"Both services failed - RapidAPI: {rapidapi_result.get('error')}, MagicHour: {magichour_result.get('error')}"
    }

def get_result_image(project_id):
    """Get the result image from MagicHour API"""
    if not MAGICHOUR_API_KEY:
        return None, "MagicHour API key not configured"
    
    try:
        headers = {
            'Authorization': f'Bearer {MAGICHOUR_API_KEY}',
        }
        
        # Use the correct get image project API
        get_url = f"https://api.magichour.ai/v1/image-projects/{project_id}"
        response = requests.get(get_url, headers=headers)
        
        if response.status_code == 200:
            result = response.json()
            status = result.get('status')
            
            if status == 'complete':
                downloads = result.get('downloads', [])
                if downloads and len(downloads) > 0:
                    return downloads[0]['url'], None
                else:
                    return None, "No download URL available"
            elif status == 'error':
                error_msg = result.get('error', 'Unknown error occurred')
                return None, f"Generation failed: {error_msg}"
            else:
                # Still processing (queued, rendering, etc.)
                return None, f"Still processing (status: {status})"
        else:
            return None, f"API request failed: {response.status_code} - {response.text}"
        
    except Exception as e:
        return None, f"Error getting result: {str(e)}"

@app.route('/')
def index():
    return render_template('index.html')

@app.route('/upload', methods=['POST'])
def upload_files():
    if 'source_image' not in request.files or 'target_image' not in request.files:
        flash('Please select both source and target images')
        return redirect(url_for('index'))
    
    source_file = request.files['source_image']
    target_file = request.files['target_image']
    
    if source_file.filename == '' or target_file.filename == '':
        flash('Please select both source and target images')
        return redirect(url_for('index'))
    
    if not (allowed_file(source_file.filename) and allowed_file(target_file.filename)):
        flash('Please upload only image files (PNG, JPG, JPEG, GIF)')
        return redirect(url_for('index'))
    
    try:
        # Generate unique filenames
        source_filename = secure_filename(f"{uuid.uuid4()}_{source_file.filename}")
        target_filename = secure_filename(f"{uuid.uuid4()}_{target_file.filename}")
        
        source_path = os.path.join(UPLOAD_FOLDER, source_filename)
        target_path = os.path.join(UPLOAD_FOLDER, target_filename)
        
        source_file.save(source_path)
        target_file.save(target_path)
        
        # Create face swap
        result = create_face_swap(source_path, target_path)
        
        if 'error' in result:
            flash(f"Error: {result['error']}")
            return redirect(url_for('index'))
        
        # Handle different service responses
        if result.get('service') == 'rapidapi':
            # RapidAPI returns immediate result
            return jsonify({
                'success': True,
                'service': 'rapidapi',
                'image_base64': result.get('image_base64'),
                'processing_time': result.get('processing_time', 0)
            })
        elif result.get('service') == 'magichour':
            # MagicHour requires polling
            return jsonify({
                'success': True,
                'service': 'magichour',
                'project_id': result['id'],
                'credits_charged': result['credits_charged']
            })
        else:
            # Unknown service or error
            flash(f"Error: Unknown service response")
            return redirect(url_for('index'))
        
    except Exception as e:
        flash(f'Error processing images: {str(e)}')
        return redirect(url_for('index'))

@app.route('/result/<project_id>')
def get_result(project_id):
    """Get the result of a face swap project"""
    result_url, error = get_result_image(project_id)
    
    if result_url:
        return jsonify({
            'success': True,
            'result_url': result_url
        })
    else:
        return jsonify({
            'success': False,
            'error': error or 'Result not ready or not found'
        })

@app.route('/status')
def status():
    """Check if API services are configured"""
    rapidapi_configured = bool(RAPIDAPI_KEY)
    magichour_configured = bool(MAGICHOUR_API_KEY)
    
    if rapidapi_configured:
        message = 'RapidAPI is configured and ready'
    elif magichour_configured:
        message = 'MagicHour API is configured as fallback'
    else:
        message = 'No API services are configured'
    
    return jsonify({
        'rapidapi_configured': rapidapi_configured,
        'magichour_configured': magichour_configured,
        'api_configured': rapidapi_configured or magichour_configured,
        'message': message
    })

if __name__ == '__main__':
    app.run(debug=True, host='0.0.0.0', port=5000, use_reloader=False)
